/*
 * 	NAME
 *      lexer - lexical scanning for vc
 *
 * 	SYNOPSIS
 *      void lex_init(char *buf);
 *          Initialise lexer input buffer and, if first call,
 *          initialises operators.
 *
 *      struct st_lex lexer();
 *      	Determine next token.  Global variable lex.sym holds token
 *      	type, lex.tok holds token content
 *
 * 	    bool lex_match(char sym);
 * 	        Moves to next symbol, returning true if current symbol is
 * 	        sym. Returns false otherwise.
 *
 * 	DESCRIPTION
 *
 *
 * 	NOTES
 * 		Single character operators stand for themselves aas
 * 		tokens. Other tokens are assigned special symbols, beginning
 * 		with S_.
 *
 * 	MODIFICATION HISTORY
 * 	Mnemonic		Rel	Date	Who
 *	lexer           1.0 020924  mpw
 *		Created.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <string.h>
#include <ctype.h>
#include "lexer.h"
#include "op.h"

/* GLOBALS */

struct st_lex lex;

/* END GLOBALS */

static char lexbuf[LEXBUFSIZ+1];
static int bufidx;
static bool ops_defined = false;

#define getlexchar() ((lexbuf[bufidx] != '\0')?lexbuf[bufidx++]:EOF)
#define ungetlexchar(ch) ((ch != '\0' && ch != EOF)?bufidx--:0)

void lex_init(char *s)
{
	strncpy(lexbuf, s, LEXBUFSIZ);
    lex.sym = S_UNDEF;
    lex.tok[0] = '\0';
	bufidx = 0;
    if (!ops_defined) {
        op_insert('=', 10);
        op_insert('&',20);
        op_insert('|',20);
        op_insert(S_EQ,30);
        op_insert('<',30);
        op_insert('>',30);
        op_insert(S_NEQ,30);
        op_insert(S_GTEQ,30);
        op_insert(S_LTEQ,30);
        op_insert('+', 40);
        op_insert('-', 40);
        op_insert('*', 50);
        op_insert('/', 50);
        op_insert('!',60);
        ops_defined = true;
    }
}

struct st_lex
lexer(void)
{
	char ch = ' ';
	int i = 0;

	lex.tok[0] = '\0';
	while (ch == ' ' || ch == '\t') ch = getlexchar();
	if (isdigit(ch)) {
		while (isdigit(ch) && i < LEXTOKSIZ) {
			lex.tok[i++] = ch;
			ch = getlexchar();
		}
		ungetlexchar(ch);
		lex.tok[i] = '\0';
		lex.sym = S_INT;
	}
	else if (isalpha(ch)) {
		while ((isalpha(ch) || isdigit(ch) || ch == '_') && i < LEXTOKSIZ) {
			lex.tok[i++] = ch;
			ch = getlexchar();
		}
		ungetlexchar(ch);
		lex.tok[i] = '\0';
		lex.sym = S_SYM;
	}
	else if (ch == '"') {
		ch = getlexchar();
		while (ch != '\n' && ch != EOF && ch != '"' && i < LEXTOKSIZ) {
			if (ch == '\\') ch = getlexchar();
			lex.tok[i++] = ch;
			ch = getlexchar();
		}
		if (ch != '"') {
			ungetlexchar(ch);
			lex.sym = S_USTR; 	/* unterminated string */
		}
		else {
			lex.sym = S_STR;
		}
		lex.tok[i] = '\0';
	}
	else if (ch == '+' || ch == '-' || ch == '*' || ch == '/' ||
			 ch == '(' || ch == ')' || ch == ';' || ch == '&' ||
             ch == '|') {
		lex.sym = ch;
		lex.tok[0] = ch; lex.tok[1] = '\0';
	}
    else if (ch == '<' || ch == '>' || ch == '!' || ch == '=') {
        lex.tok[0] = ch;
        ch = getlexchar();
        if (ch == '=') {
            switch (lex.tok[0]) {
                case '<':
                    lex.sym = S_LTEQ;
                    break;
                case '>':
                    lex.sym = S_GTEQ;
                    break;
                case '!':
                    lex.sym = S_NEQ;
                    break;
                case '=':
                    lex.sym = S_EQ;
            }
            lex.tok[1] = ch;
            lex.tok[2] = '\0';
        }
        else {
            ungetlexchar(ch);
            lex.sym = lex.tok[0];
            lex.tok[1] = '\0';
        }
    }
	else if (ch == EOF) {
		lex.sym = S_EOF;
		strcpy(lex.tok,"end-of-file");
	}
	else if (ch == '\n') {
		lex.sym = S_EOE;
		strcpy(lex.tok,"newline");
	}
	else {
		/* unrecognised token */
		while (ch != '\n' && ch != EOF) {
			lex.tok[i++] = ch;
			ch = getlexchar();
		}
		lex.tok[i] = '\0';
		ungetlexchar(ch);
		lex.sym = S_UNDEF;
	}
    return lex;
}

bool
lex_match(char ch)
{
	if (ch == lex.sym) {
		(void) lexer();
        return true;
	}
    return false;
}
